#!/usr/bin/perl

use lib './install_util';
use GuessDirectories;
use File::Basename qw(&dirname);
use File::Copy;
use Cwd;
use FindBin '$Bin';
require 5.005003;

my $VERSION = '1.03';

my %INSTALL_FILES = (
                     'CONF'   => [
                                  'conf/01.locuslink.conf',
                                  'conf/locuslink.xview.xml',
                                 ],
                     'SCRIPT' => [
                                  'bin/bulk_load_xrt.PLS',
                                 ],
                     'HTDOCS' => [
                                  'htdocs',
                                 ],
                     'CGIBIN' => [
                                  'cgi-bin/tbrowse.PLS',
                                  'cgi-bin/xview.PLS',
                                 ],
                     'LIB'    => [
                                  'lib',
                                 ],
                    );


my @OPTIONS = (keys %INSTALL_FILES, 'PREFIX');
my $OPTIONS = join ('|',@OPTIONS);
my %OPTIONS;

my @argv = @ARGV;
@ARGV = ();

foreach (@argv) {
  if (/^--?h/i) {
    die <<END;
To customize install locations, provide one or more of the options
PREFIX, CONF, HTDOCS, CGIBIN, or SCRIPT as in CONF=/usr/local/apache/conf.

For RedHat Linux, you will want to use:
    perl Install.PL CONF=/etc/httpd/conf \\
                     HTDOCS=/var/www/html \\
                     CGIBIN=/var/www/cgi-bin \\
                     SCRIPT=/usr/bin \\
                     LIB=/usr/local/lib/site_perl

For most versions of Apache built from source code, use:
    perl Install.PL PREFIX=/usr/local/apache SCRIPT=/usr/bin

END
;
  }

  elsif (/($OPTIONS)=(.+)/og) {
    $OPTIONS{$1} = interpolate($2);
  }

  else {
    push @ARGV,$_;
  }
}


print STDERR "\nThis program will guide you step by step to install BioXRT system.\n\n";

if (-e 'BioXRT.def' && !%OPTIONS) {
  print STDERR "\nShould I use your previous settings to set the file paths and options (y/n) [y]? ";
  chomp (my $line = <>);
  if (!$line || $line =~ /^[yY]/) {
    print STDERR "Using previous settings for file paths and options.\n";
    open F,'BioXRT.def' or die "BioXRT.def: $!";
    while (<F>) {
      chomp;
      next if /^\#/;
      next unless /^($OPTIONS)=(.+)/o;
      $OPTIONS{$1} = interpolate($2);
    }
    close F;
  }
}

if ($OPTIONS{PREFIX} && !(defined $OPTIONS{CONF})) {
  $OPTIONS{CONF} = "$OPTIONS{PREFIX}/conf";
}
if ($OPTIONS{PREFIX} && !(defined $OPTIONS{HTDOCS})) {
  $OPTIONS{HTDOCS} = "$OPTIONS{PREFIX}/htdocs";
}
if ($OPTIONS{PREFIX} && !(defined $OPTIONS{CGIBIN})) {
  $OPTIONS{CGIBIN} = "$OPTIONS{PREFIX}/cgi-bin";
}

# set hard-coded values
$OPTIONS{PREFIX} ||= '/usr/local/apache';
unless ($OPTIONS{CONF}) {
  $OPTIONS{CONF} = GuessDirectories->conf || "$OPTIONS{PREFIX}/conf";
}
unless ($OPTIONS{HTDOCS}) {
  $OPTIONS{HTDOCS} = GuessDirectories->htdocs || "$OPTIONS{PREFIX}/htdocs";
}
unless ($OPTIONS{CGIBIN}) {
  $OPTIONS{CGIBIN} = GuessDirectories->cgibin || "$OPTIONS{PREFIX}/cgi-bin";
}
unless ($OPTIONS{SCRIPT}) {
  $OPTIONS{SCRIPT} = GuessDirectories->bin || '/usr/local/bin';
}
unless ($OPTIONS{LIB}) {
  $OPTIONS{LIB} = GuessDirectories->lib || '/usr/local/lib/site_perl';
}

print STDERR "\nWhere is the htdocs/documentroot directory for Apache? [$OPTIONS{HTDOCS}]: ";
chomp (my $htdocs_dir = <>);
if ($htdocs_dir) { $OPTIONS{HTDOCS} = interpolate($htdocs_dir); }
unless (-d $OPTIONS{HTDOCS}) { die "Directory '$OPTIONS{HTDOCS}' does not exist\n"};

print STDERR "\nIn which directory do you want to keep the BioXRT configuration? [$OPTIONS{CONF}]: ";
chomp (my $conf_dir = <>);
if ($conf_dir) { $OPTIONS{CONF} = interpolate($conf_dir); }
unless (-d $OPTIONS{CONF}) { die "Directory '$OPTIONS{CONF}' does not exist\n"};

print STDERR "\nWhere is the cgi-bin directory for Apache? [$OPTIONS{CGIBIN}]: ";
chomp (my $cgi_dir = <>);
if ($cgi_dir) { $OPTIONS{CGIBIN} = interpolate($cgi_dir); }
unless (-d $OPTIONS{CGIBIN}) { die "Directory '$OPTIONS{CGIBIN}' does not exist\n"};

print STDERR "\nIn which directory do you want to put the executable files (scripts)? [$OPTIONS{SCRIPT}]: ";
chomp (my $bin_dir = <>);
if ($bin_dir) { $OPTIONS{SCRIPT} = interpolate($bin_dir); }
unless (-d $OPTIONS{SCRIPT}) { die "Directory '$OPTIONS{SCRIPT}' does not exist\n"};

print STDERR "\nIn which directory do you want to install the BioXRT perl libraries? [$OPTIONS{LIB}]: ";
chomp (my $lib_dir = <>);
if ($lib_dir) { $OPTIONS{LIB} = interpolate($lib_dir); }
unless (-d $OPTIONS{LIB}) { die "Directory '$OPTIONS{LIB}' does not exist\n"};
unless (grep { $_ eq $OPTIONS{LIB} } @INC) {
  my $res = "";
  while (!($res =~ /^[nNyY]/)) {
    print STDERR "Directory $OPTIONS{LIB} is not in your current Perl include paths, do you still want to continue? (yes/no): ";
    chomp ($res = <>);
  }
  if ( $res =~ /^[nN]/ ) {
    die "\nUser cancelled the installation!\n\n";
  } else {
    print STDERR "\n***************** NOTE ***********************\n";
    print STDERR "*\n";
    print STDERR "* To make BioXRT work, you may have to add the following line in some scripts:\n";
    print STDERR "* use lib '$OPTIONS{LIB}'\;\n";
    print STDERR "*";
    print STDERR "\n**********************************************\n";
  }
};


# workaround for known bug in FindBin library
$Bin = "." if -f './Install.PL' && -T './Install.PL';

open F,">$Bin/BioXRT.def" or die "Can't open $Bin/BioXRT.def for writing: $!";
print F "# This hints file contains configuration information used by the installation of BioXRT\n\n";
foreach (keys %OPTIONS) {
  print F "$_=$OPTIONS{$_}\n";
}
print F "VERSION=$VERSION\n";
close F;

print STDERR <<END;

----------------------------------------------
            File Path Settings

END
;
foreach (@OPTIONS) {
  printf STDERR "%7s %s\n",$_,$OPTIONS{$_} unless ($_ eq 'PREFIX');
}
print STDERR <<END;

(Run perl Install.PL -h for help on changing)
----------------------------------------------

END
;


print STDERR "Should I start to install the BioXRT system using the above settings (y/n) [y]? ";
chomp (my $line = <>);
unless (!$line || $line =~ /^[yY]/) { die "User cancelled the installation!\n\n"; }


foreach (keys %INSTALL_FILES) {

  my $err = 0;
  my $type = $_;
  my @files = @{$INSTALL_FILES{$type}};
  my $inst_dir = $OPTIONS{$type};
  
  print STDERR "\nInstall $type into $inst_dir ...... \n";

  my $source_dir = "";

  foreach (@files) {
    my $file = $_;
    $source_dir = dirname($file);

    if ($file =~ /\.PL$/ or $file =~ /\.PLS$/) {
      my $status = system("perl $file");
      die "unable to extract $file\nInstallation failed!\n" if $status;
    }
  }

  if ($type eq 'LIB') {    # install libs
    my $status = system("cp -r ./$files[0]/* $inst_dir");
    die "unable to copy libraries to $inst_dir\nInstallation failed!\n" if $status;

  } elsif ($type eq 'HTDOCS') {    # install htdocs
    $inst_dir .= '/bioxrt';
    mkdir($inst_dir, 0777) or die "unable to create $inst_dir directory\nInstallation failed!\n" unless (-d $inst_dir);

    my $status = system("cp -r ./$files[0]/* $inst_dir");
    die "unable to copy documents to $inst_dir\nInstallation failed!\n" if $status;

  } else {    # install other files
    my @cp_files = ();
    if ($source_dir) {
      opendir (DIR, $source_dir) or die "unable to open $source_dir\nInstallation failed!\n";
      @cp_files = readdir(DIR);
      closedir (DIR);
    }

    if ($type eq 'CONF') {
      $inst_dir .= '/bioxrt.conf';
      mkdir($inst_dir, 0777) or die "unable to create $inst_dir directory\nInstallation failed!\n" unless (-d $inst_dir);
    }

    foreach (@cp_files) {
      next if /README/;
      next if /CVS/;
      next if /^[.\#]/;
      next if /~$/;
      next if /\.PLS$/;
      next unless (-f "$source_dir/$_");

      my $source_file = "$source_dir/$_";
      my $inst_file = "$inst_dir/$_";

      my $confirm = 1;

      if ($type eq 'CONF' and -f $inst_file) {  # before copy conf file, test if file exists or not
        $confirm = 0;
        my $res = "";
        while (!($res =~ /^[nNyY]/)) {
          print STDERR "Config file $_ already exists, do you want to overwrite it? (yes/no): ";
          chomp ($res = <>);
        }
        $confirm = 1 if ( $res =~ /^[yY]/ );
      }

      if ($confirm) {
        copy("$source_file","$inst_file") or die "unable to copy to $inst_file\nInstallation failed!\n";
        if ($type =~ /SCRIPT|CGIBIN/) {
          chmod 0755, "$inst_file" or die "unable to make $inst_file world executable\nInstallation failed!\n";
        }
      }
    }

  }

  print STDERR "Done!\n"
}
print STDERR "\n";

## checking some pre-required modules
my @warn = ();
eval {
  require XML::Parser;
};
push(@warn, "  Warning: Unable to find the pre-required XML::Parser module! CGI script xview may not work.") if $@;
eval {
  require XML::Parser::EasyTree;
};
push(@warn, "  Warning: Unable to find the pre-required XML::Parser::EasyTree module! CGI script xview may not work.") if $@;

print STDERR "BioXRT system installed ";
unless (@warn) {
  print STDERR "successfully!\n\n";
} else {
  print STDERR "with warning message(s):\n";
  print STDERR join("\n",@warn), "\n\n";
}


# check the MySQL version, has to be higher than 4.0
my $mysql = `mysql -V`;
if ($mysql =~ /Distrib\s+(.+?),/){
  my $ver = $1;
  unless ($ver ge '4.0') {
    print STDERR "Warning: the version of the MySQL Server running on your system is $ver, lowest version required is 4.0!\n\n";
  }
}


#### subrountines
sub interpolate {
  my $path = shift;
  my ($to_expand,$homedir);
  return $path unless $path =~ m!^~([^/]*)!;
  eval {
    if ($to_expand = $1) {
      $homedir = (getpwnam($to_expand))[7];
    } else {
      $homedir = (getpwuid($<))[7];
    }
    return $path unless $homedir;

    $path =~ s!^~[^/]*!$homedir!;
  };
  return $path;
}

