# $Id: ConfigFile.pm,v 1.6 2004/11/02 17:20:29 junjun Exp $

=head1 NAME

Bio::ConfigFile - Provides the access to the configuration file

=head1 SYNOPSIS

	use Bio::ConfigFile;
	
	# Open the configuration file
	my $conf = Bio::ConfigFile->new($conf_file);

	# Get value of an attribute
	my $value = $conf->getAttribute($attrib_name, $stanza);


=cut


package Bio::ConfigFile;
use strict;

=head1 Methods

=cut

=head2 new

 Title   : new
 Usage   : $config = ConfigFile->new($file_name);
 Function: Creates a new ConfigFile object from the file name
 Returns : A ConfigFile object
 Args    : The name of the config file

=cut

sub new {
    my $class = shift;
    my $self = {};
    bless($self, $class);

    $self->_init(shift);

    return $self;
}

=head2 _init

 Title   : _init
 Usage   : $config->_init($file_name);
 Function: Initializes the object at construction time. (Include parsing)
 Returns : None
 Args    : The config file name
 Note    :
 Example :

=cut

sub _init {
    my $self = shift;
    my $fileName = shift;

    # storage of attribute values for object
    $self->{general} = {};
    $self->{stanza} = {};

    # open the file and parse each line
    open(IN, $fileName) || die "Cannot open config file $fileName: $!";

    # variables to help in parsing
    my $readingAttribute = 0; # false
    my $currentAttribute = "";
    my $currentStanza = "";

    while(my $line = <IN>) {
        chomp($line);

        if ($line =~ /^#/ or $line =~ /^\s*$/) { # is a comment line
            # ignore comment and empty lines
        } elsif ($line =~ /^\[(.+)\]/) { # is a stanza definition
            $currentStanza = $1;
            push @{$self->{stanzas}}, $currentStanza;
        } elsif ($line =~ /^(\S+.*?)\s*?=(.*)/) { # beginning of an attribute
            $currentAttribute = $1;
            $self->_addAttribute($currentStanza, $currentAttribute, $2);
            push @{$self->{_attributes}{$currentStanza}}, $currentAttribute;
        } elsif ($line =~ /^\s+/){ # an attribute value
            $self->_addAttribute($currentStanza, $currentAttribute, $line);
        }
    }

    close(IN);
}

=head2 _addAttribute

 Title   : _addAttribute
 Usage   : $config->_addAttribute($stanza, $attribute, $value);
 Function: Adds the attribute to the object's hash
 Returns : None
 Args    : The stanza the attribute belongs in (empty string if it is in the general scope)
           The attribute name
           The attribute value
 Throws  :
 Note    :
 Example :

=cut

sub _addAttribute {
    my $self = shift;
    my $stanza = shift;
    my $attribute = shift;
    my $value = shift;

    # trim value
    $value =~ s/^\s+//;
    $value =~ s/\s+$//;
    $value =~ s/^'(.*)'$/$1/;

    if ($stanza) { # belongs to a stanza
        if ($self->{stanza}{$stanza}{$attribute}) { # previous value exists
            $self->{stanza}{$stanza}{$attribute} .= " " . $value;
        } else { # first time
            $self->{stanza}{$stanza}{$attribute} = $value;
        }
    } else { # does not belong to a stanza
        if ($self->{general}{$attribute}) { # previous value exists
            $self->{general}{$attribute} .= " " . $value;
        } else { # first time
            $self->{general}{$attribute} = $value;
        }
    }
}

=head2 getAttribute

 Title   : getAttribute
 Usage   : $value = $config->getAttribute($attribute_name, $stanza);
 Function: Retrieves the attribute within the stanza specified
 Returns : The value of the attribute within the stanza specified
 Args    : 
           The attribute name
           The stanza name, if stanza omit means the general
 Throws  :
 Note    :
 Example :

=cut

sub getAttribute {
    my $self = shift;
    my $key = shift;
    my $stanza = shift;

    unless ($stanza) {
        return $self->{general}{$key};
    }

    if ($self->{stanza}{$stanza}) { # stanza exists
        return $self->{stanza}{$stanza}{$key};
    }
    return undef;
}

=head2 stanzas

 Title   : stanzas
 Usage   : @value = $config->stanzas();
 Function: Retrieves all the stanzas
 Returns : The array of all stanzas
 Args    : None
 Throws  :
 Note    :
 Example :

=cut

sub stanzas {
    my $self = shift;
    return (ref($self->{stanzas}) eq 'ARRAY') ? @{$self->{stanzas}} : ();
}


=head2 attributes

 Title   : attributes
 Usage   : @value = $config->attributes($stanza);
 Function: Retrieves the name of all attributes within the specified stanza
 Returns : The array of the name of attributes within the specified stanza
 Args    : stanza or None, if no arg provided return the attributes of the general part
 Throws  :
 Note    :

=cut

sub attributes {
    my $self = shift;
    my $stanzas = shift;
    $stanzas ||= "";
    return (ref($self->{_attributes}{$stanzas}) eq 'ARRAY') ? @{$self->{_attributes}{$stanzas}} : ();
}



=head2 asCode

 Title   : asCode
 Usage   : $returnValue = ConfigFile->asCode($config->getAttribute("key1", "stanza1"), arg1, arg2, ...);
 Function: This is a STATIC CLASS method. It should not be invoked with an object.
           It evaluates the string representation of the code with the given arguments.
 Returns : Whatever is returned by the code
 Args    : The code to be executed (as string). It should be enclosed in "sub {}"
           Any argument(s) necessary for the execution of the code.
 Throws  :
 Note    :
 Example :

=cut

sub asCode {
    my $class = shift;
    my $codeString = shift;

    my $code = eval $codeString;
    return &$code(@_);
}

1;
__END__

=head1 AUTHOR

This module was initially written by Terence Tang.

Now it's maintained by Junjun Zhang.

Copyright (c) 2004 The Centre for Applied Genomics.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
