# $Id: XRT.pm,v 1.12 2004/11/09 21:45:22 junjun Exp junjun $

=head1 NAME

Bio::DB::XRT -- Storage and retieval of XRT (Cross-Referenced Tables) database

=cut

=head1 SYNOPSIS

Interface to the XRT database  

=cut

package Bio::DB::XRT;

use strict;
use DBI;
use Bio::Root::Root;
use Bio::DB::GFF::Util::Rearrange;                # for rearrange() (bioperl)
use Bio::DB::GFF::Adaptor::dbi::caching_handle;   # (bioperl)
use Bio::DB::XRT::View;                           # a set of query results from XRT database

use vars qw($VERSION @ISA);

$VERSION = '1.0';
@ISA = qw(Bio::Root::Root);
##############################################################################

=head1 Methods


=cut

=head2 new

 Title   : new
 Usage   : $xrt = Bio::DB::XRT->new(@args)
 Function: create a new XRT object
 Returns : a Bio::DB::XRT object
 Args    : see below
 Status  : Public

 This is the constructor for the XRT object.  It is called automatically
 by Bio::DB::XRT-E<gt>new.  The following class-specific arguments are recgonized:

   Argument       Description
   --------       -----------

    -dsn           the DBI data source, e.g. 'dbi:mysql:myxrt'

    -user          username for authentication

    -pass          the password for authentication

=cut

# Create a new Bio::DB::XRT object
sub new {
  my $class = shift;
  my ($xrt_db,$username,$auth,$other) = rearrange([
                                                    [qw(XRTDB DB DSN)],
                                                    [qw(USERNAME USER)],
                                                    [qw(PASSWORD PASS)],
                                                  ],@_);

  $xrt_db  || $class->throw("new(): Provide a data source or DBI database");

  if (!ref($xrt_db)) {
    my $dsn = $xrt_db;
    my @args;
    push @args,$username if defined $username;
    push @args,$auth     if defined $auth;
    $xrt_db = Bio::DB::GFF::Adaptor::dbi::caching_handle->new($dsn,@args)
       || $class->throw("new(): Failed to connect to $dsn: "
                        . Bio::DB::GFF::Adaptor::dbi::caching_handle->errstr);
  } else {
    $xrt_db->isa('DBI::db') 
       || $class->throw("new(): $xrt_db is not a DBI handle");
  }

  ### fill in partial object
  my $object = bless { dbh => $xrt_db }, $class;

  return $object;
}


=head2 dbh

 Title   : dbh
 Usage   : $dbh = $xrt->dbh
 Function: get the XRT database handle
 Returns : a DBI handle
 Args    : none
 Status  : Public

This method may also be called xrt_db

=cut

sub dbh    { shift->{dbh} }
sub xrt_db { shift->{dbh} }


=head2 class2cid

 Title   : class2cid
 Usage   : %class2cid = $xrt->class2cid(@args)
 Function: get the hash %class2cid which coverts class into cid, return the 
           class id if class name provided
 Returns : hash %class2cid or class id
 Args    : class name (optional)
 Status  : Public

=cut

sub class2cid {
  my $self = shift;
  my $class = shift;
  my %class2cid = $self->classes();
  return defined($class) ? $class2cid{$class} : %class2cid;
}

=head2 cid2class

 Title   : cid2class
 Usage   : %cid2class = $xrt->cid2class(@args)
 Function: get the hash %cid2class which coverts class id into class name, return the
           class name if class id provided
 Returns : hash %cid2class or class name
 Args    : class id (optional)
 Status  : Public

=cut

sub cid2class {
  my $self = shift;
  my $cid = shift;
  my %class2cid = $self->classes();
  my %cid2class = reverse %class2cid;
  return defined($cid) ? $cid2class{$cid} : %cid2class;
}

=head2 attribute2aid

 Title   : attribute2aid
 Usage   : %attribute2aid = $xrt->attribute2aid(@args)
 Function: get the hash %attribute2aid which coverts attribute into aid, return the
           attribute id if attribute name provided
 Returns : hash %attribute2aid or attribute id
 Args    : attribute name (optional)
 Status  : Public

=cut

sub attribute2aid {
  my $self = shift;
  my $attribute = shift;
  my %attribute2aid = $self->attributes();
  return defined($attribute) ? $attribute2aid{$attribute} : %attribute2aid;
} 

=head2 aid2attribute

 Title   : aid2attribute
 Usage   : %aid2attribute = $xrt->aid2attribute(@args)
 Function: get the hash %aid2attribute which coverts attribute id into attribute, return the
           attribute if attribute id provided
 Returns : hash %aid2attribute or attribute name
 Args    : attribute id (optional)
 Status  : Public

=cut

sub aid2attribute {
  my $self = shift;
  my $aid = shift;
  my %attributes = $self->attributes();
  my %aid2attribute = reverse %attributes;
  return defined($aid) ? $aid2attribute{$aid} : %aid2attribute;
}

=head2 view

 Title   : view
 Usage   : $view = $xrt->view(@args)
 Function: create a view object 
 Returns : new view object
 Args    : see below
 Status  : Public

This method uses the following arguments:

   Argument       Description
   --------       -----------
    -factory       


=cut

sub view {
  my $self = shift;
  my $view =  Bio::DB::XRT::View->new(-factory => $self, @_);
  return $view;
}

=head2 classes

 Title   : classes
 Usage   : $db->classes
 Function: return hash of classes
 Returns : a hash of classes
 Args    : none
 Status  : public

=cut

sub classes {
  my $self = shift;
  my $query = $self->make_classes_query or return;
  my $sth           = $self->dbh->do_query($query);
  my %classes = ();
  while (my ($cid,$class) = $sth->fetchrow_array) {
     $classes{$class}=$cid;
  }
  $sth->finish;
  return %classes;
}

=head2 make_classes_query

 Title   : make_classes_query
 Usage   : $query = $db->make_classes_query
 Function: return query fragment for generating list of classes
 Returns : a query and args
 Args    : none
 Status  : public

=cut

sub make_classes_query {
  my $self = shift;
  my $query = "SELECT cid,class FROM gclass";
  return $query;
}

=head2 attributes

 Title   : attributes
 Usage   : $db->attributes
 Function: return ref of hash of attributes in database
 Returns : hash of attributes
 Args    : class name (optional)
 Status  : public

=cut

sub attributes {
  my $self = shift;
  my $class = shift;
  my ($query, @args) = $self->make_attributes_query($class) or return;
  my $sth           = $self->dbh->do_query($query, @args);
  my %attributes = ();
  while (my ($aid,$attribute) = $sth->fetchrow_array) {
     $attributes{$attribute}=$aid;
  }
  $sth->finish;
  return %attributes;
}

=head2 make_attributes_query

 Title   : make_attributes_query
 Usage   : $query = $db->make_attributes_query
 Function: return query fragment for generating hash of attributes
 Returns : a query and args
 Args    : none
 Status  : public

=cut

sub make_attributes_query {
  my $self = shift;
  my $class = shift;
  my @args = ();
  push @args, $class if defined $class;
  my $query = defined $class ? 
                               "SELECT DISTINCT a.* FROM gattribute a, gdata b, gclass c WHERE a.aid=b.aid AND b.cid = c.cid AND c.class = ?" :
			       "SELECT aid,attribute FROM gattribute";
  return ($query, @args);
}


=head2 schema

 Title   : schema
 Usage   : $schema = $db->schema
 Function: return the CREATE script for the schema
 Returns : a list of CREATE statemetns
 Args    : none
 Status  : protected

This method returns a list containing the various CREATE statements
needed to initialize the database tables.

=cut

sub schema {
  my %schema = (
                gdata =>{ 
table=> q{
CREATE TABLE gdata (
  gid                   int(11)       NOT NULL auto_increment,
  gref                  varchar(100)  NOT NULL,
  cid                   int(11)       NOT NULL,
  aid                   int(11)       NOT NULL,
  gvalue                text          NOT NULL,
  PRIMARY KEY (gid),
  KEY (gvalue(255)),
  FULLTEXT KEY gfulltext (gvalue),
  KEY gentry (gref,cid),
  KEY (gref),
  KEY (cid),
  KEY (aid),
  UNIQUE KEY nonr (gref,cid,aid,gvalue(255))
) TYPE=MyISAM
}  # gdata table
}, # gdata

                gclass => {
table=> q{
CREATE TABLE gclass (
  cid                  int(11)        NOT NULL auto_increment,
  class                varchar(255)   NOT NULL,
  PRIMARY KEY (cid),
  UNIQUE KEY (class)
) TYPE=MyISAM
}  # gclass table
}, # gclass

                gattribute => {
table=> q{
CREATE TABLE gattribute (
  aid                  int(11)        NOT NULL auto_increment,
  attribute            varchar(255)   NOT NULL,
  PRIMARY KEY (aid),
  KEY (attribute)
) TYPE=MyISAM
}  # gattribute table
}, # gattribute

                grelationship => {
table=> q{
CREATE TABLE grelationship (
  pcid                 int(11)        NOT NULL,
  pgref                varchar(100)   NOT NULL,
  ccid                 int(11)        NOT NULL,
  cgref                varchar(100)   NOT NULL,
  UNIQUE KEY nonr (pcid,pgref,ccid,cgref),
  KEY (pcid),
  KEY (pgref),
  KEY (ccid),
  KEY (cgref),
  KEY pentry (pgref,pcid),
  KEY centry (cgref,ccid)
) TYPE=MyISAM
}  # grelationship table
}, # grelationship

);

  return \%schema;
}

=head2 db_initialize

 Title   : db_initialize
 Usage   : $success = $db->db_initialize($drop_all)
 Function: initialize the database
 Returns : a boolean indicating the success of the operation
 Args    : a boolean indicating whether to delete existing data
 Status  : protected

This method will load the schema into the database.  If $drop_all is
true, then any existing data in the tables known to the schema will be
deleted.

Internally, this method calls schema() to get the schema data.

=cut

# Create the schema from scratch.
# You will need create privileges for this.
sub db_initialize {
  my $self = shift;
  my $erase = shift;
  $self->drop_all if $erase;

  my $dbh = $self->dbh;
  my $schema = $self->schema;
  foreach my $table_name ($self->db_tables) {
    my $create_table_stmt = $schema->{$table_name}{table} ;
    $dbh->do($create_table_stmt) ||  warn $dbh->errstr;
  }
  1;
}

=head2 db_tables

 Title   : db_tables
 Usage   : @tables = $db->db_tables
 Function: return list of tables that belong to this module
 Returns : list of tables
 Args    : none
 Status  : protected

This method lists the tables known to the module.

=cut

# return list of tables that "belong" to us. 
sub db_tables {
  my $schema = shift->schema;
  return keys %$schema;
}

=head2 drop_all

 Title   : drop_all
 Usage   : $db->drop_all
 Function: empty the database
 Returns : void
 Args    : none
 Status  : protected

This method drops the tables known to this module.  Internally it
calls the abstract tables() method.

=cut

# Drop all the XRT tables -- dangerous!
sub drop_all {
  my $self = shift;
  my $dbh = $self->dbh;
  my $schema = $self->schema;

  local $dbh->{PrintError} = 0;
  foreach ($self->db_tables) {
    $dbh->do("drop table $_"); # || warn $dbh->errstr;
  }
}


=head2 simple_query

=cut

sub simple_query {
  my $self = shift;
  my ($cid,$class,$aid,$attribute,$gref) = rearrange([
                                                 [qw(CID CLASS_ID)],
                                                 [qw(CLASS)],
                                                 [qw(AID ATTRIBUTE_ID PID PROPERTY_ID)],
                                                 [qw(ATTRIBUTE PROPERTY)],
                                                 [qw(GREF REFID)],
                                               ],@_);
  $self->throw("simple_query(): Need class ($class) or class id ($cid)") if (!$cid && !$class);
  $self->throw("simple_query(): Need attribute or attribute id") if (!$aid && !$attribute);

  $cid ||= $self->class2cid($class);
  $aid ||= $self->attribute2aid($attribute);

  my ($query, @args) = $self->make_query_for_simple_query($cid,$aid,$gref);
  my $sth  = $self->dbh->do_query($query,@args);
  my @results = ();
  while (my ($gvalue) = $sth->fetchrow_array) {
    push @results, $gvalue;
  }
  $sth->finish;
  return wantarray ? @results : $results[0];
}

=head2 make_query_for_simple_query

=cut

sub make_query_for_simple_query {
  my ($self,$cid,$aid,$gref) = @_;
  my @args = ($cid,$aid);
  my $query = "";
  if (defined $gref) {
    push @args, $gref;
    $query = "SELECT gvalue FROM gdata WHERE cid = ? AND aid = ? AND gref = ? ORDER BY gid";
  } else {
    $query = "SELECT gvalue FROM gdata WHERE cid = ? AND aid = ? ORDER BY gid";
  }

  return ($query,@args);
}


=head2 relational_query

 Args     : see below

   Argument       Description
   --------       -----------

    -mcid          main class ID

    -mclass        main class

    -mrefids       the array of the main refs

    -a2show        this is actually the sittings of the view, contains the attributes to be shown

=cut

sub relational_query {
  my $self = shift;
  my %view_results = ();
  my ($mcid,$mclass,$mrefids,$cid,$class,$aid,$attribute,$a2show) = rearrange([
                                                 [qw(MCID MCLASS_ID)],
                                                 [qw(MCLASS MAIN_CLASS)],
                                                 [qw(MREFID MREFIDS MREF)],
                                                 [qw(CID)],
                                                 [qw(CLASS)],
                                                 [qw(AID PID)],
                                                 [qw(ATTRIBUTE PROPERTY)],
                                                 [qw(A2SHOW)],
                                               ],@_);

  $mcid ||= $self->class2cid($mclass);
  return unless $mcid;

  ######### when mrefids is not provided means all the entries in the main class, but returns () while $mrefids references to a empty array
  return $a2show ? %view_results : wantarray ? () : undef if (ref $mrefids eq 'ARRAY' and @{$mrefids} == 0);

  my $mrefs = ref($mrefids) ? "'" . join("','", @{$mrefids}) . "'" : "'$mrefids'";

  my ($cids, $aids);
  my %cids = ( $mcid=>1 );
  my %aids = ();
  my $selfClassQuery = 0;
  my $relatedClassQuery = 0;

  if ($a2show) {
    foreach (@{$a2show}) {
      unless ($_->[2] =~ /^\s*sub/) {
        $selfClassQuery = 1 if ($_->[1] == 0);
        $relatedClassQuery = 1 if ($_->[1]);
        $cids{$_->[1]} = 1;
        $aids{$_->[2]} = 1;
      }
    }
    $cids = join(',', sort{$a <=> $b}(keys(%cids)));
    $aids = join(',', sort{$a <=> $b}(keys(%aids)));
  } else {
    $cid = 0 if ($class eq '0' or $class eq '-');
    $cid = $self->class2cid($class) unless defined($cid);
    $cid = -1 if ($class && $cid eq undef);
  
    $aid ||= $self->attribute2aid($attribute);
    $aid = -1 if ($attribute && !($aid));

    if ($cid == -1 or $aid == -1) {
      return;
    } else {
      $cids = $cid;
      $aids = $aid;
      $cids ? $relatedClassQuery = 1 : $selfClassQuery = 1;
    }
  }

  my $query;
  my @args = ($mcid);
  my @sth;

  if ($selfClassQuery) {
    $query = $self->make_relational_query(0, $mrefs, $cids, $aids);
    push @sth, $self->dbh->do_query($query,@args) if $query;
  }   

  if ($relatedClassQuery) {
    for (my $i = 1; $i <= 2; $i++) {
      $query = $self->make_relational_query($i, $mrefs, $cids, $aids);
      push @sth, $self->dbh->do_query($query,@args) if $query;
    }
  }

  my @values = ();
  foreach (@sth) {
    while (my ($mgref, $gid, $gref, $cid, $aid, $gvalue) = $_->fetchrow_array) {
      push @{ $view_results{$mgref}->{"$gref~-~$cid~-~$aid"} },  $gvalue;
      push @values, $gvalue;
    }
    $_->finish;
  }

  return $a2show ? %view_results : wantarray ? @values : $values[0];

}


=head2 make_relational_query

=cut

sub make_relational_query {
  my ($self, $qtype, $mrefs, $cids, $aids) = @_;

  $mrefs = ($mrefs eq "\'\'") ? "not in ( $mrefs )" : " in ( $mrefs )";

  $self->throw("make_relational_query(): qtype must be 0 or 1 or 2") unless ($qtype=~/[012]/);
  my $query = $qtype == 0 ? "SELECT gref, gid, gref, 0 cid, aid, gvalue FROM gdata WHERE cid = ? AND gref $mrefs AND aid in ( $aids ) " :
              $qtype == 1 ? "SELECT DISTINCT a.cgref, b.* FROM grelationship AS a , gdata AS b  WHERE (a.pcid = b.cid AND a.pgref = b.gref AND ccid = ? AND cgref $mrefs AND cid in ( $cids ) AND aid in ( $aids ) ) " :
              $qtype == 2 ? "SELECT DISTINCT a.pgref, b.* FROM grelationship AS a , gdata AS b  WHERE (a.ccid = b.cid AND a.cgref = b.gref AND pcid = ? AND pgref $mrefs AND cid in ( $cids ) AND aid in ( $aids ) ) " :
              "";
  return $query;
}


=head2 query_main_gref_by_keyword

 Args     : see below

   Argument       Description
   --------       -----------

    -mcid          main class ID

    -mclass        main class

    -keyword       the keyword want to search for

    -qtype         the type of query: 0=>self, 1=>parent, 2=>child

=cut

sub query_main_gref_by_keyword {
  my $self = shift;
  my ($mcid,$mclass,$rcid,$rclass,$aid,$attribute,$keyword,$filter,$exact) = rearrange([
                                                 [qw(MCID MCLASS_ID)],
                                                 [qw(MCLASS MAIN_CLASS)],
                                                 [qw(RCID RCLASS_ID)],
                                                 [qw(RCLASS RELATED_CLASS)],
                                                 [qw(AID ATTRIBUTE_ID PID PROPERTY_ID)],
                                                 [qw(ATTRIBUTE PROPERTY)],
                                                 [qw(KEYWORD SEARCH)],
                                                 [qw(FILTER)],
                                                 [qw(EXACT)],
                                               ],@_);
  $self->throw("query_by_keyword(): Need main class or main class id") if (!$mcid and !$mclass);
  $self->throw("query_by_keyword(): Need the keyword or reference of filter") unless (defined($keyword) or ref($filter));

  $mcid ||= $self->class2cid("$mclass");
  $rcid = $self->class2cid("$rclass") if ($rclass and $rcid eq undef);
  $rcid = -1 if ($rcid eq undef and defined($rclass));
  $aid ||= $self->attribute2aid("$attribute") if ($attribute and $aid eq undef);
  $aid =  -1 if ($aid eq undef and $attribute);

  my @results = ();

  if (defined($keyword) and !ref($filter)) { # keyword searching
    my $j = 0;
    my $k = 3;
    if ($rcid eq '0') {
      $j = 0;
      $k = 1;
    } elsif ($rcid=~/^\d+$/ and $rcid > 0) {
      $j = 1;
      $k = 3;
    }
    my %done = ();
    for (my $qtype = $j; $qtype < $k; $qtype++) {
      my $query = $self->make_query_by_keyword($qtype,$keyword,$exact,$rcid,$aid);
      my @args = ($mcid);
      my $sth  = $self->dbh->do_query($query,@args);
      while (my ($gref) = $sth->fetchrow_array) {
        push @results, $gref unless $done{$gref}++;
      }
      $sth->finish;
    }
    @results = sort @results;

  } elsif (!defined($keyword) and ref($filter)) { # filtering
#   data structure of filter: [[$rcid,$aid,$keyword,1],   ["or"] ]
    my @filters = @{$filter};
    my $opt = pop @filters if (@filters > 1);
    my $op = 'AND';
    $op = 'OR' if ($opt eq undef || lc($opt->[0]) eq 'or' || @filters==1);

    return () unless (@filters > 0);

    if (uc($op) eq 'AND') {   
      my @grefs = ();
      my $ii = 0;
      foreach (@filters) {
        my $rcid = $_->[0];
        $rcid = $self->class2cid($rcid) unless ($rcid =~ /^(-*\d+)$/);
        my $qtype = $rcid == 0 ? 0 : 12;
	my @query = $ii==0 ? $self->make_query_by_filter($qtype, $op, [$_]) : $self->make_query_by_filter($qtype, $op, [$_], \@grefs);
        @grefs = ();
        foreach my $query (@query) {
          my $sth = $self->dbh->do_query($query,$mcid);
	  my $gref;
          if (ref($sth)) {
            while (($gref) = $sth->fetchrow_array) {
              push @grefs, $gref;
	    }
            $sth->finish;
          }
        }
       last unless @grefs;
	$ii++;
      }
      @results = sort @grefs;

    } else {
      my %res = ();
      my %done = ();

      for (my $qtype = 0; $qtype < 3; $qtype++) {
        my $rtype = $qtype == 0 ? 0 : 1;
        my @query = $self->make_query_by_filter($qtype, $op, \@filters);
        my $count = 0;
        foreach my $query (@query) {
          my $sth;
          if ($query) {
            $sth  = $self->dbh->do_query($query,$mcid);
            $count++;
          }
          if (ref($sth)) {
            while (my ($gref) = $sth->fetchrow_array) {
              $done{$gref}++;
            }
            $sth->finish;
          }
        }
  
        foreach (keys %done) {
          $res{$_}++ if $done{$_} == $count;
        }
      }

      @results = sort(keys(%res));
    }

  } elsif (defined($keyword) and ref($filter)) {  # in this stuation, call this method itself separately with keyword and filter
    my @results1 = $self->query_main_gref_by_keyword( -mcid=>$mcid, -rcid=>$rcid, -aid=>$aid, -keyword=>$keyword, -exact=>$exact );
    my @results2 = $self->query_main_gref_by_keyword( -mcid=>$mcid, -filter=>$filter, -exact=>$exact );
    my %unique = map {$_=>1} @results1;
    foreach (@results2) {
      $unique{$_}++;
    }

    foreach (sort(keys(%unique))) {
      push @results, $_ if $unique{$_} > 1;
    }
  } else {
    $self->throw("query_by_keyword(): Need the keyword or reference of filter");
  }

  return @results;
}


=head2 make_query_by_keyword

=cut

sub make_query_by_keyword {
  my $self = shift;
  my $qtype = shift;
  my $keyword = shift;
  my $exact = shift;
  my $rcid = shift;
  my $aid  = shift;

  $keyword =~ s/'/\\'/g;
  $keyword =~ s/"/\\"/g;
#  $keyword =~ s/\(/\\\(/g;
#  $keyword =~ s/\)/\\\)/g;
  $keyword = "\"$keyword\"" if ($exact and !($keyword =~ /\*+/));

  $self->throw("make_query_by_refid(): qtype must be 0 or 1 or 2") unless ($qtype=~/[012]/);
  my $query = $qtype == 0 ? "SELECT DISTINCT gref FROM gdata WHERE cid = ? AND ".($exact ? " match gvalue against ('$keyword' in boolean mode) " : " gvalue rlike '$keyword' ").($aid > 0 ? " AND aid = '$aid'" : "") :
              $qtype == 1 ? "SELECT DISTINCT cgref FROM grelationship AS a , gdata AS b  WHERE (a.pcid = b.cid AND a.pgref = b.gref AND ccid = ? AND ".($exact ? " match gvalue against ('$keyword' in boolean mode) " : " gvalue rlike '$keyword' ")." ) ".($rcid=~/^(\d+|-\d+)$/ ? " AND cid = $rcid" : "").($aid=~/^(\d+|-\d+)$/ ? " AND aid = $aid" : "")." ORDER BY gid" :
              $qtype == 2 ? "SELECT DISTINCT pgref FROM grelationship AS a , gdata AS b  WHERE (a.ccid = b.cid AND a.cgref = b.gref AND pcid = ? AND ".($exact ? " match gvalue against ('$keyword' in boolean mode) " : " gvalue rlike '$keyword' ")." ) ".($rcid=~/^(\d+|-\d+)$/ ? " AND cid = $rcid" : "").($aid=~/^(\d+|-\d+)$/ ? " AND aid = $aid" : "")." ORDER BY gid" :
              "";
  return $query;
}

=head2 make_query_by_filter

=cut

sub make_query_by_filter {
  my ($self,$qtype,$op,$filters,$grefs) = @_;
  my @filters = @$filters;

  if (defined $grefs) {
    my @grefs = ref($grefs) eq "ARRAY" ? @$grefs : ();
    $grefs = "'".join("', '", @grefs)."'";
  }

  my @sqlfragment0 = ();
  my @sqlfragment1 = ();
  foreach (@filters) {
    return "" unless (ref($_) eq 'ARRAY');
    my $rcid = $_->[0];
    $rcid = $self->class2cid("$rcid") unless ($rcid =~ /^(\d+|-\d+)$/);
	return if ($rcid eq "");
    my $aid  = $_->[1];
    $aid = $self->attribute2aid("$aid") unless ($aid =~ /^(\d+|-\d+)$/);
	return if ($aid eq "");
    my $keyword = $_->[2];
    return "" unless defined($keyword);
    $keyword =~ s/'/\\'/g;
    $keyword =~ s/"/\\"/g;
    $keyword =~ s/\(/\\\(/g;
    $keyword =~ s/\)/\\\)/g;
    my $compare = $_->[3];
    $compare ||= 'rlike';
    return "" unless ($compare=~/^(match|rlike|in|=|>|>=|<|<=|<>|regexp)$/i);
    my $new_keyword = $keyword;
    $new_keyword = "'$keyword'" unless ($keyword=~/^(\d+\.\d*|\d*\.\d+|\d+)$/ and !($compare=~/^(rlike|regexp)$/i));
    if (lc($compare) eq 'in') {
       $new_keyword = "";
       my @keywords = split /\,|\s+/, $keyword;
       foreach (@keywords) {
          s/^\s+//;
          s/\s+$//;
          $new_keyword .= "'$_', ";
       }
       chop $new_keyword;
       chop $new_keyword;
       $new_keyword = "($new_keyword)";
    } elsif (lc($compare) eq 'match') {
      $new_keyword =~ s/^'/'"/ unless $new_keyword =~ /\*+/;
      $new_keyword =~ s/'$/"'/ unless $new_keyword =~ /\*+/;
    }

    if ($qtype == 0 and $rcid == 0) {
      my $sqlF = (lc($compare) eq 'match') ? "( aid = $aid AND match gvalue against ($new_keyword in boolean mode))" : "( aid = $aid AND gvalue $compare $new_keyword )";
      push @sqlfragment0, $sqlF;
    } elsif ($qtype > 0 and $rcid > 0) {
      my $sqlF = (lc($compare) eq 'match') ? "( cid = $rcid AND aid = $aid AND match gvalue against ($new_keyword in boolean mode))" : "( cid = $rcid AND aid = $aid AND gvalue $compare $new_keyword )";
      push @sqlfragment1, $sqlF;
    }
  }

  my @query = ();
  if (uc($op) eq "OR") {
    my $sqlfrag = $qtype == 0 ? join(" $op ", @sqlfragment0) : join(" $op ", @sqlfragment1);
    $sqlfrag = "AND ( $sqlfrag )" if ($sqlfrag);
  
    my $query = ($qtype == 0 and $sqlfrag) ? "SELECT DISTINCT gref FROM gdata WHERE cid = ? $sqlfrag" :
                ($qtype == 1 and $sqlfrag) ? "SELECT DISTINCT cgref FROM grelationship AS a , gdata AS b  WHERE a.pcid = b.cid AND a.pgref = b.gref AND ccid = ? $sqlfrag" :
                ($qtype == 2 and $sqlfrag) ? "SELECT DISTINCT pgref FROM grelationship AS a , gdata AS b  WHERE a.ccid = b.cid AND a.cgref = b.gref AND pcid = ? $sqlfrag" : "";

    push @query, $query;
  } else {
    my @sqlfrag = $qtype == 0 ?  @sqlfragment0 : @sqlfragment1;
    my @qtype = $qtype == 12 ? (1, 2) : ($qtype) ;
    foreach (@qtype) {
      my $qtype = $_;
      foreach my $sqlfrag (@sqlfrag) {
        my $query = ($qtype == 0 and $sqlfrag) ? "SELECT DISTINCT gref FROM gdata WHERE cid = ? AND $sqlfrag" :
                  ($qtype == 1 and $sqlfrag) ? "SELECT DISTINCT cgref FROM grelationship AS a , gdata AS b  WHERE a.pcid = b.cid AND a.pgref = b.gref AND ccid = ? AND $sqlfrag" :
                  ($qtype == 2 and $sqlfrag) ? "SELECT DISTINCT pgref FROM grelationship AS a , gdata AS b  WHERE a.ccid = b.cid AND a.cgref = b.gref AND pcid = ? AND $sqlfrag" : "";
        $query .= " AND ".($qtype == 0 ? "gref" : $qtype == 1 ? "cgref" : "pgref")." in ( $grefs ) " if $grefs;
        push @query, $query;
      }
    }
  }

  return @query;
}


=head2 DESTORY

=cut

sub DESTORY {
  my $self = shift; 
  $self->dbh->disconnect if defined $self->dbh;
}


1;

__END__

=head1 AUTHOR

Junjun Zhang E<lt>junjun@genet.sickkids.on.caE<gt>.

Copyright (c) 2004 The Centre for Applied Genomics.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.  See DISCLAIMER for
disclaimers of warranty.

=cut
