# $Id: TBrowser.pm,v 1.13 2008/08/28 15:19:05 junjun Exp $

package Bio::DB::XRT::TBrowser;

use strict;

use Bio::ConfigFile;
use Carp qw(carp croak);
use File::Basename 'basename';
use Bio::DB::XRT;


use vars qw($VERSION);

$VERSION = '1.03';

use constant DEFAULT_WIDTH => '100%';
use constant DEFAULT_ROWS => 50;
use constant DEFAULT_OUTPUT => 'hHtml';
use constant ROWS_TO_CHOOSE => '20 50 100 200 0';

################################################################

=head2 new

  Title       : new
  usage       : Bio::DB::XRT::TBrowser->new(@args)
  Function    :
  Returns     :
  Args        :
  Status      :

=cut

sub new {
  my $class = shift;
  my $conf_dir = shift;
  my $safe = shift;

  croak("$conf_dir: not a directory") unless -d $conf_dir;

  my $self = bless { conf_dir=>$conf_dir, safe=>$safe }, $class;

  $self->_init();

  $self;
}

=head2 _init


=cut

sub _init {
  my $self = shift;
  my $conf_dir = $self->{conf_dir};

  opendir(D,$conf_dir) or croak "Couldn't open $conf_dir: $!";
  my @conf_files = map { "$conf_dir/$_" } grep {/\.conf$/} readdir(D);
  close D;

  # get modification times
  my %mtimes     = map { $_ => (stat($_))[9] } @conf_files;

  my %done = ();
  my $ii = 0;
  for my $file (sort {$a cmp $b} @conf_files) {
    $ii++;
    my $basename = basename($file,'.conf');
    $basename =~ s/^\d+\.//;
    next if defined($self->{conf}{$basename}{mtime})
      && ($self->{conf}{$basename}{mtime} >= $mtimes{$file});
    my $config = Bio::ConfigFile->new($file) or next;
    $self->{conf}{$basename}{data}  = $config;
    $self->{conf}{$basename}{mtime} = $mtimes{$file};
    push @{$self->{sources}}, $basename unless $done{$basename}++;
    $self->{source} ||= $basename;
  }
  croak("No conf files in the directory: $conf_dir") unless $ii;
  $self->{xrt_db} = Bio::DB::XRT->new($self->db_settings);
  $self->_get_viewconfs;
  1;
}

sub sources {
  my $self = shift;
  return ref($self->{sources}) ? @{$self->{sources}} : ();
}

sub source {
  my $self = shift;
  if (@_) {
    my $source = shift;
    if ($self->{conf}{$source}) {
      $self->{source} = $source;
      $self->xrt_db->DESTORY if defined $self->xrt_db;
      $self->{xrt_db} = Bio::DB::XRT->new($self->db_settings);
      $self->_get_viewconfs;
      return $self->{source};
    } else {
      return;
    }
  } else {
    return $self->{source};
  }
}

sub xrt_db {
  shift->{xrt_db};
}

sub tables {
  my $self = shift;
  my $source = $self->{source};

  my @keys = $self->{conf}{$source}{data}->stanzas;

  my @ret = grep { !(/^_/) } @keys; # hide the tables whose name started with _

  return wantarray ? @ret : $ret[0];
}

sub table {
  my $self = shift;
  my $table = shift;

  return $self->{table} unless defined($table);

  my @tables = $self->tables;
  if (grep { $_ eq $table} @tables) {
    $self->{table} = $table;
  } else {
    my $default_table = $self->default_table;
    warn "The specified table '$table' doesn't exist in source '".$self->source."', default table '$default_table' has been chosen.";
    $self->{table} = $default_table;
  }
  $self->_set_view;
  $self->rstart(1);
  return $self->{table};
}

sub default_table {
  my $self = shift;
  my $default_table = $self->setting('default table', 'general');
  my @all_tables = $self->tables;
  if (grep {$_ eq $default_table} @all_tables) {
    return $default_table;
  } else {
    warn "In source ".$self->source.", default table setting is wrong or omitted, switched to the first available table '$all_tables[0]'.";
    $all_tables[0] ? return $all_tables[0] : croak "No tables available in source: ".$self->source;
  }
}

sub default_cols {
  my $self = shift;
  my $cols = $self->setting('columns');
  my @cols = split /\s+/, $cols;
  return @cols;
}

sub width {
  my $self = shift;
  my $input = shift;
  if ($input) {
    return $self->{width} = $input;
  }
  my $width = $self->setting('width') || $self->setting('width','general');
  $width = DEFAULT_WIDTH unless ($width =~ /^\d+%*$/);
  return $self->{width} = $width;
}

sub viewconfs {
    return shift->{viewconfs};
}

sub _set_view {
  my $self = shift;
  my $viewid = $self->setting('viewid') || $self->setting('view_id');
  my $viewconf = $self->viewconfs->{$viewid};
  if ($viewid and $viewconf){
    my $view = $self->xrt_db->view(-viewconf=>$viewconf, -output=>$self->output);
    return $self->{view} = $view;
  } else {
    croak "Error in the definition of table whose viewid is '$viewid' (in source: ".$self->source.")";
  }
}

sub view {
  return shift->{view};
}

=head2 _get_viewconfs

=cut
sub _get_viewconfs {
  my $self = shift;
  my $xrt_db = $self->xrt_db;
  my %class2cid = $xrt_db->class2cid;
  my %attribute2aid = $xrt_db->attribute2aid;

  my $config = $self->{conf}{$self->source}{data};

  my %viewconfs = ();
  my @tableKeys = $config->stanzas;
  foreach (@tableKeys) {
    my $tableKey = $_;
    my $viewid = $config->getAttribute("viewid", $tableKey) || $config->getAttribute("view_id", $tableKey);
    next unless defined $viewid;
    my $mainClass = $config->getAttribute("mclass", $tableKey) || $config->getAttribute("main_class", $tableKey);
    my $mcid = $class2cid{$mainClass};
    next unless defined $mcid;

    my @options = $config->attributes($tableKey);
    my @columnIDs = ();
    my %colDef = ();
    my %unique = ();
    foreach (@options) {
        if (/^(.+?)\.(.*)$/) {
            my ($columnID,$field) = ($1, $2);
            push (@columnIDs, $columnID) unless $unique{$columnID}++; 
            if ($field =~ /^rclass$/i) {
                my $value = $config->getAttribute($_, $tableKey);
                my $rcid = $value eq "0" ? "0" : $class2cid{$value};
                $colDef{$columnID}->[0] = $rcid if defined $rcid; 
            } elsif ($field =~ /^(attribute|property)$/i) {
                my $value = $config->getAttribute($_, $tableKey);
                my $aid = $attribute2aid{$value} ? $attribute2aid{$value} : $value;
                $colDef{$columnID}->[1] = $aid if defined $aid; 
            } elsif ($field =~ /^header$/i) {
                my $value = $config->getAttribute($_, $tableKey);
                $colDef{$columnID}->[2] = $value;
            } elsif ($field =~ /^(url|link)$/i) {
                my $value = $config->getAttribute($_, $tableKey);
                $colDef{$columnID}->[3] = $value;
            } elsif ($field eq undef) {
                my $value = $config->getAttribute($_, $tableKey);
                my ($rclass, $attribute, $header, $url) = split /\\t|::/, $value;
                my $rcid = $rclass eq "0" ? "0" : $class2cid{$rclass};
                my $aid = $attribute2aid{$attribute} ? $attribute2aid{$attribute} : $attribute;
                $colDef{$columnID}->[0] = $rcid;
                $colDef{$columnID}->[1] = $aid;
                $colDef{$columnID}->[2] = $header;
                $colDef{$columnID}->[3] = $url;
            }
        }
    }

    my @aids = ();
    foreach (@columnIDs) {
        my $rcid = $colDef{$_}->[0];
        my $aid = $colDef{$_}->[1];
        my $header = $colDef{$_}->[2];
        my $url = $colDef{$_}->[3];
        push @aids, [$_, $rcid, $aid, $header, $url] if (defined($rcid) and ($aid=~/\d+/ or $aid=~/^\s*sub\s*\{.*\}\s*$/));
    }

    $viewconfs{$viewid} = { mcid=>$mcid, a2show=>\@aids } if @aids;
  }

  $self->{viewconfs} = \%viewconfs;
}

sub output {
  my $self = shift;
  my $arg = shift;

  if (defined $arg) {
    return $self->{output} = $arg;
  } else {
    my $output = $self->{output};
    $output ||= $self->setting('layout');
    $output ||= $self->setting('layout','general');
    $output ||= DEFAULT_OUTPUT;
    return $self->{output} = $output;
  }
}

sub set_rows {
  my $self = shift;
  my $input = shift;
 
  my $max_rows = $self->max_rows;
  $max_rows ||= 0;

  if (defined($input)) {
    return $self->{rows} = ($input <= $max_rows or $max_rows ==0) ? $input : $max_rows if ($input=~/^\d+$/);
    return -1;
  }

  my $rows = $self->setting('rows');
  $rows = $self->setting('rows','general') unless ($rows=~/^\d+$/);
  $rows = DEFAULT_ROWS unless ($rows=~/^\d+$/);
  
  return $self->{rows} = ($rows <= $max_rows or $max_rows ==0) ? $rows : $max_rows;
}

sub rows {
  my $self = shift;
  $self->set_rows unless ($self->{rows} =~ /^\d+$/);
  return $self->{rows};
}

sub max_rows {
  my $self = shift;
  return $self->setting('max_rows');
}

sub rstart {
  my $self = shift;
  my $input = shift;
  if (defined $input) {
    $self->{rstart} = $input if ($input =~ /^\d+$/ and $input > 0);
  } 
  $self->{rstart} = 1 if $self->{rstart} eq undef;
  return $self->{rstart};
}

sub mrefids {
  my $self = shift;
  my $xrt_db = $self->xrt_db; 

  my $filter = $self->filter;

  my $exact = 1;  ## by default, use full text indexing match which is very fast
  my $keyword = $self->keyword;
  if (ref($keyword) eq 'ARRAY') {
    $exact = $keyword->[1] eq 'match' ? 1 : 0; #### TODO: we should sort this out later, probably we should use operators
    $keyword = $keyword->[0];
  }

#  if ($keyword=~/^(.+):(.*)?/) {
#    $keyword = $1;
#    $exact = 1 if ($2 eq 'match');  
#  }

  my $mclass = $self->setting('mclass') || $self->setting('main_class');
  my $mcid = $xrt_db->class2cid("$mclass");

  return () unless ($xrt_db and defined($mcid));

  my @mrefids = ();

  my $mfilter = $self->setting('mfilter');  # table's main filter
  if (ref($mfilter) eq 'CODE') {
    $mfilter = &$mfilter; 
    @mrefids = $xrt_db->query_main_gref_by_keyword(-mcid=>$mcid, -filter=>$mfilter);
    if (defined $keyword or $filter) {
      my @f_mrefids = $xrt_db->query_main_gref_by_keyword(-mcid=>$mcid, -keyword=>$keyword, -filter=>$filter, -exact=>$exact);
      my %unique = map {$_=>1} @mrefids;
      @mrefids = ();
      foreach (@f_mrefids) { 
        $unique{$_}++;
      }
      foreach (keys(%unique)) {
        push @mrefids, $_ if $unique{$_} > 1;
      }
      @mrefids = sort(@mrefids);
    }
  } else {
    if (defined $keyword or $filter) {
      my @f_mrefids = $xrt_db->query_main_gref_by_keyword(-mcid=>$mcid, -keyword=>$keyword, -filter=>$filter, -exact=>$exact);
      @mrefids = sort(@f_mrefids);
    } else {
    # if no mfilter set means all records of the main class
      @mrefids = $xrt_db->simple_query(-cid=>$mcid, -attribute=>'ID');
    }
  }

  $self->{mrefids} = \@mrefids;

  return wantarray ? @mrefids : (@mrefids+0);
}

sub url {
  my $self = shift;
  my $url = shift;
  return $self->{url} = $url if (defined $url);
  return $self->{url};
}

sub keyword {
  my $self = shift;
  my $keyword = shift;
  return $self->{keyword} = $keyword if (defined $keyword);
  return $self->{keyword};
}

sub filter {
  my $self = shift;
  my $filter = shift;

  if (defined($filter)) {
    return $self->{filter} = $filter if (ref($filter) eq 'ARRAY' and @{$filter} > 0);
    return -1;
  }

  return $self->{filter};
}

sub current_mrefids {
  my $self = shift;
  my @mrefids = ref $self->{mrefids} ? @{$self->{mrefids}} : $self->mrefids;

  my $total_row = (@mrefids + 0);
  my $rnum = $self->rows;

  if ($rnum eq '0') {  # means all
    $self->rstart(1);
    $self->set_rows($total_row);
    return @mrefids;
  }

  my $rstart = $self->rstart;
  if ($rstart > @mrefids) {
    $rstart = $self->rstart(1);
  }

  my $rend = $rstart + $rnum - 1;
  if ($rend > @mrefids) {
    $rend = (@mrefids+0);
#    $self->set_rows($rend-$rstart+1);
  }

  @mrefids[$rstart-1..$rend-1];
}

sub adjust_cols {
  my $self =shift;
  $self->adjust_view_a2show(@_);
}

sub adjust_view_a2show {
  my $self = shift;
  my @cols = @_;

  my $view = $self->view;

  return unless (@cols >0);
  return unless (ref $view && $view->can('adjust_a2show'));
  return unless $view->adjust_a2show(@cols);

  1;
}

sub page_header {
  shift->setting('page header','general');
}

sub show_back {
  shift->setting('show back','general');
}

sub setting {
  my $self = shift;
  my $sname = shift;
  my $stanza = shift;
  my $source = shift;

  my $table = $self->{table};
  $stanza ||= $table;

  $stanza = undef if (lc($stanza) eq 'general');

  $source ||= $self->{source};

  my $setting = $self->{conf}{$source}{data}->getAttribute($sname, $stanza);

  return $setting unless ($setting =~ /^\s*sub\s*\{/ and $self->{safe});
  my $coderef = eval $setting;
  warn $@ if $@;
  return $coderef;
}


=head2 db_settings()

  (@argv) = $conf->db_settings;

Returns the appropriate arguments for connecting to Bio::DB::XRT.  It
can be used this way:

  $xrt_db = Bio::DB::XRT->new($conf->db_settings);

=cut

# get database adaptor name and arguments
sub db_settings {
  my $self = shift;
  my $cmdl = shift;

  my $DSN = $self->setting("xrt_db", "general") || $self->setting("gaf_db", "general");
  my $HOST = $self->setting("host", "general");
  my $PORT = $self->setting("port", "general");
  my $USER = $self->setting("user", "general");
  my $PASSWORD = $self->setting("pass", "general");
  
  my (@argv,$AUTH);

  if (defined $USER) {
    push @argv,(-user=>$USER);
    $AUTH .= " -u$USER";
  }
  if (defined $PASSWORD) {
    push @argv,(-pass=>$PASSWORD);
    $AUTH .= " -p$PASSWORD";
  }
  if (defined $HOST) {
    $AUTH .= " -h$HOST";
  }
  
  $DSN = "dbi:mysql:$DSN" if !ref($DSN) && $DSN !~ /^(?:dbi|DBI):/;
  my $dsn = $DSN.(($HOST) ? ":$HOST" : "").(($PORT) ? ":$PORT" : "");
  push @argv,(-dsn=>$dsn);

  defined($cmdl) ? $AUTH : @argv;
}

sub rows2choose {
  my $self = shift;

  my @all_rows = ($self->setting('rows2choose'), 
              $self->setting('rows2choose','general'),
              ROWS_TO_CHOOSE) ;

  my @newrows = ();

  OUT: foreach (@all_rows) {
    my @rows = split /\s+/;
    next unless @rows;
    my $hasZero = 0;
    my $error = 0;
    @newrows = ();
    foreach (sort {$a <=> $b} @rows) {
      if (/^\d+$/) {
        if ($_ > 0) {
          push @newrows, $_;
        }else{
          $hasZero = 1;
        }
      } else {
        $error = 1;
        next OUT;
      }
    }    
    push @newrows, 0 if $hasZero;
    last unless $error;
  }

  return @newrows;
}


sub navigation_bar {
  my $self = shift;

  my $source = $self->source;
  my $table = $self->table;
  my $url = $self->url;

  my $randID = int(rand(1000000));

  my $row_start = $self->rstart;
  my $rows_per_page = $self->rows;
  my $max_rows = $self->max_rows;
  my @mrefids = @{$self->{mrefids}};  
  my $total_rows = (@mrefids+0);  
  my @rows2choose = $self->rows2choose;

  # add the user set row into the array
  unless (grep {/^$rows_per_page$/} @rows2choose) {
    push @rows2choose, $rows_per_page;
    @rows2choose = sort {$a <=> $b} @rows2choose;
    if (grep {/^0$/} @rows2choose) {
      shift @rows2choose;
      push @rows2choose, 0;
    }
  }

  # get rid of the rows greater than max_rows or total_rows, remove row=0 as well if $total_rows > $max_rows
  my @newrows = ();
  foreach (@rows2choose) {
    push @newrows, $_ unless ( $_ >= $total_rows or ($max_rows and ($_ > $max_rows or  ($_ == 0 and $total_rows > $max_rows))) );
  }

  # calculate the total page
  my $tmp1 = $rows_per_page==0 ? 0 : $total_rows / $rows_per_page;
  my $tmp2 = $tmp1 ; $tmp2 =~ s/\.\d*$//;
  my $total_page = ($tmp1 eq $tmp2) ? $tmp2 : $tmp2 + 1;

  # calculate the current page
  my $tmp1 = $rows_per_page==0 ? 0 : $row_start / $rows_per_page;
  my $tmp2 = $tmp1 ; $tmp2 =~ s/\.\d*$//;
  my $current_page = ($tmp1 eq $tmp2) ? $tmp2 : $tmp2 + 1;

  # some other variables
  my $first_rstart = 1;
  my $next_rstart = $row_start + $rows_per_page;
  my $back_rstart = $row_start - $rows_per_page;
  my $last_rstart = ($total_page - 1) * $rows_per_page + 1;


########## output the navigation bar

  print "<table width='850'><tbody><tr class='tab_control'>\n";
  
  print "<form name='pages$randID' onsubmit='document.location.href=document.pages$randID.page[document.pages$randID.page.selectedIndex].value'>
   <td width='14%'>Go page:
   <select class='tablesox' name='page' onchange='document.location.href=document.pages$randID.page[document.pages$randID.page.selectedIndex].value'>\n";
  for(my $i=1; $i<=$total_page; $i++){  
    print "<option ".($i == $current_page ? "selected " : "")."value='$url?source=$source&table=$table&rnum=$rows_per_page&rstart=".(($i-1)*$rows_per_page+1)."'>$i</option>\n";
  }
  print "</select>\n</td></form>\n";
  
  print "<td width='32%' align='left'><b>";
    if ($back_rstart >= 1 and $total_page > 1) {
      print "<span class='ctrl_en_button'>\n";
      print "<<<a class='ctrl_en_button' href='$url?rnum=$rows_per_page&rstart=1'>First</a>\n" ;
      print "<<a class='ctrl_en_button' href='$url?source=$source&table=$table&rnum=$rows_per_page&rstart=$back_rstart'>Previous</a>\n";
      print "</span> ";
    } else {
      print "<span class='ctrl_dis_button'>\n";
      print "<<<u>First</u>\n";
      print "<<u>Previous</u>\n";
      print "</span>\n";
    }
  
    if ($next_rstart <= $total_rows and $total_page > 1) {
      print "<span class='ctrl_en_button'>\n";
      print "<a class='ctrl_en_button' href='$url?source=$source&table=$table&rnum=$rows_per_page&rstart=$next_rstart'>Next</a>>\n";
      print "<a class='ctrl_en_button' href='$url?source=$source&table=$table&rnum=$rows_per_page&rstart=$last_rstart'>End</a>>>\n";
      print "</span> ";
    } else {
      print "<span class='ctrl_dis_button'>\n";
      print "<u>Next</u>>\n";
      print "<u>End</u>>>\n";
      print "</span>\n";
    }
  print "</b></td>";
  
  print "
   <form name='tables$randID' onsubmit='document.location.href=document.tables$randID.url[document.tables$randID.url.selectedIndex].value'>
   <td width='20%'>
   <select class='tablesox' name='url' onchange='document.location.href=document.tables$randID.url[document.tables$randID.url.selectedIndex].value'>";
  foreach (@newrows) {
    if ($_ == 0) {
      print "<option ".($total_page == 1 ? "selected " : "")."value='$url?source=$source&table=$table&rnum=0'>All</option>\n";
    } else {
      print "<option ".($rows_per_page == $_ ? "selected " : "")."value='$url?source=$source&table=$table&rnum=$_'>$_</option>\n";
    }
  }
  print "</select>
   records per page</td>
   </form>
  ";
  
  print "<td width='34%' align='right'>Record: ".($total_rows==0?"":$row_start)." to ".( $total_rows==0?"":($row_start+$rows_per_page-1>$total_rows?$total_rows:$row_start+$rows_per_page-1) )." \(Total: ".($total_rows)."\)</td>\n";
  
  print "</tr></tbody></table>\n";
  

##############


}

sub show_table {
  my $self = shift;
  my $tableOnly = shift;
  my $output = $self->output;
  $self->view->output($output) if $output;

  my $isFilter = $self->filter;

  my $keyword = $self->keyword;
  $keyword = ref($keyword) eq 'ARRAY' ? $keyword->[0] : $keyword;

  my $view = $self->view;

  # invoke current_mrefids, then set the mrefid
  my @c_mrefids = $self->current_mrefids;
  $view->mrefid(\@c_mrefids);

  if ($output =~ /^(hHtml|vHtml)$/i and !$tableOnly) { # only html outputs need navigation bar

    my $table_title = $self->setting('title');
    print "<h3>$table_title</h3>\n";

    print " Keyword: <font color='red'>$keyword</font>" if length($keyword)>0;
    print " Filter: <font color='red'>On</font>" if ref $isFilter;

    unless (@c_mrefids) {
      my $nomatchwarn = $self->setting('no match warn')
                     || $self->setting('no match warn', 'general')
                     || "Sorry, no match records found!";
      print "<br><br> <font size='5' color='red'>$nomatchwarn</font><br><br>\n";
      return;
    }

    my $width = $self->width;

    # header of table
    print "\n<table border='0' cellpadding='2' cellspacing='0' width='$width'>\n<tr><td>";
    # invoke navigation_bar
    $self->navigation_bar;
  }

  # output the real table
  print $view->show_view($tableOnly ? "" : $self->rstart);

  # close the table
  if ($output =~ /^(hHtml|vHtml)$/i and !$tableOnly) {
    # invoke navigation_bar
    $self->navigation_bar;

    my $footer = $self->setting('footer');
    print "</td></tr></table>\n".($footer ? "<br>Note: $footer" : "")."<br>\n" ;

    print "<p><a href='#top'>Back to the top!</a></p>\n";
    print "<p><i>This database is implemented using the <a href='http://projects.tcag.ca/bioxrt'>BioXRT</a> platform.</i><br><font size='2'>BioXRT table browser version $VERSION</font></p>\n";
  }

}



=head2 DESTORY

=cut

sub DESTORY {
	my $self = shift; 
	$self->xrt_db->DESTORY if defined $self->xrt_db;
}


1;

=head1 AUTHOR

Junjun Zhang E<lt>junjun@genet.sickkids.on.caE<gt>.

Copyright (c) 2004 The Centre for Applied Genomics.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.  See DISCLAIMER for
disclaimers of warranty.

=cut
