# $Id: View.pm,v 1.7 2008/08/28 14:22:48 junjun Exp $

=head1 NAME
Bio::DB::XRT::view -- Storage and retieval of generic annotation data

=cut

package Bio::DB::XRT::View;

use strict;
use DBI;
use Bio::Root::Root;
use Bio::DB::GFF::Util::Rearrange; # for rearrange()
use URI::Escape('uri_escape'); # for url encoding

use vars qw($VERSION @ISA);

$VERSION = '1.0';
@ISA = qw(Bio::Root::Root);
##############################################################################


=head2 new

 Title   : new
 Usage   : $v = Bio::DB::XRT::View->new(@args)
 Function: create a new XRT View
 Returns : a new Bio::DB::XRT::View object
 Args    : see below
 Status  : Public

This method creates a new Bio::DB::XRT::View object.  Generally
this is called automatically by the Bio::DB::XRT module and
derivatives.

This function uses a named-argument style:

 -factory or -xrt     a Bio::DB::XRT to use for database access
 -viewconf            conf of the view 
 -mclass              main class
 -mref                ref id of entry in the main class


=cut

sub new {
  my $package = shift;
  my ($factory,$viewconf,$mclass,$mcid,$mrefid,$a2show,$output)=
    rearrange([
               [qw(FACTORY XRT XRTDB)],
               [qw(VIEWCONF VIEW_CONF)],
               [qw(MCLASS MAIN_CLASS)],
               [qw(MCID MCLASS_ID MAIN_CLASS_ID)],
               [qw(REFID GREF MREFID MREFIDS)],
               'A2SHOW',
               [qw(OUTPUT OUT)],
              ],@_);

  $package = ref $package if ref $package;
  $factory or $package->throw("new(): provide a -factory argument");
  my $outdated = 1;

  my $object = bless { factory => $factory }, $package;

  ############ fill in the properties of the object

  $mcid ||= $factory->class2cid($mclass);

  @{$object}{qw(mcid mrefid outdated)}
    = ($mcid, $mrefid, $outdated);

  $object->viewconf($viewconf) if ($viewconf);

  $object->a2show($a2show) if ($a2show);

  $object->output($output) if ($output);

  return $object;
}



# read-only accessors

=head2 factory

 Title   : factory
 Usage   : $s->factory
 Function: get the xrt object
 Returns : a Bio::DB::XRT
 Args    : none
 Status  : Public

This is a read-only accessor for the Bio::DB::XRT object

=cut

sub factory { shift->{factory} }
sub xrt { shift->{factory} }


=head2 viewconf

=cut

sub viewconf {
  my $self = shift;
  my $viewconf = shift;
  if ($viewconf) {
    $self->{mcid} = $viewconf->{mcid};
    $self->a2show($viewconf->{a2show});
    $self->{outdated} = 1;
  }
}

sub mcid {
  my $self = shift;
  my $mcid = shift;
  if ($mcid) {
    my $mclass = $self->factory->cid2class($mcid);
    return unless defined $mclass;
    $self->{mcid} = $mcid;
    $self->{a2show} = undef;
    $self->{outdated} = 1;
  }
  $self->{mcid};
}

sub mclass {
  my $self = shift;
  my $mclass = shift;

  if ($mclass) {
    my $mcid = $self->factory->class2cid($mclass);
    return unless $mcid;
    $self->{mcid} = $mcid;
    $self->{a2show} = undef;
    $self->{outdated} = 1;
  }

  my $mcid = $self->{mcid};
  $mcid ||= -1;

  $self->factory->cid2class($mcid);
}

sub mrefid {
  my $self = shift;
  my $mrefid = shift;
  if ($mrefid) {
    $self->{mrefid} = $mrefid;
    $self->{outdated} = 1;
  }
  $self->{mrefid};
}

sub a2show {
  my $self = shift;
  my $a2show = shift;
  if (ref($a2show)) {
    $self->{a2show} = $a2show;
    
    my @headers = ();
    my %toheaders = ();
    foreach (@{$a2show}) {
      my $ci = $_->[0];
      my $header = $_->[3] || $self->xrt->aid2attribute($_->[2]);
      push @headers, $header unless ($_->[0] =~ /^_/);
      $toheaders{"$ci"} = $header;
    }
    $self->{headers} = \@headers;
    $self->{toHeader} = \%toheaders;

    $self->{outdated} = 1;
  }
  $self->{a2show};
}

sub adjust_a2show {
  my $self = shift;
  my @new_colids = @_;
  return -1 unless (@new_colids);

  my $a2show = $self->a2show;
  return -1 unless ($a2show);

  my %a2show_colid = ();
  foreach (@{$a2show}) {
    $a2show_colid{$_->[0]} = $_;
  }

  my @new_a2show;
  foreach (grep {$_ =~ /^_.*$/} keys(%a2show_colid)) {  # put the internal cols (begin with '_')back in the new a2show
    push @new_a2show, $a2show_colid{$_};
  }

  foreach (@new_colids) {
    push @new_a2show, $a2show_colid{$_} if (ref $a2show_colid{$_});
  }

  return $self->a2show(\@new_a2show);
}

sub col2value {
  my $self = shift;
  my $colid = shift;
  my $value = "";
  my @values= ();
  if (defined($colid)) {
    my %show = %{$self->{col2value}};
    $value = ref($show{$colid}) ? $show{$colid}->[0]->[0] : "";
    @values = ref($show{$colid}) ? @{ $show{$colid} } : ();
  }
  return wantarray ? @values : $value;
}

sub columns {
  my $self = shift;

  my $a2show = $self->{a2show};
  my @a2show = @{$a2show} if ref($a2show);
  return unless(@a2show);

  my $cols = {};
  foreach (@a2show) {
#    unless ($_->[0] =~ /^_/) {
      my $ci = $_->[0];
      push @{$cols->{cols}}, $ci;
      $cols->{settings}{$ci}{header} = $self->header($ci);
      $cols->{settings}{$ci}{rcid} = $_->[1];
      $cols->{settings}{$ci}{aid} = $_->[2];
      $cols->{settings}{$ci}{calc} = 1 unless ($_->[2]=~/^\d+$/);  # aid is not a number means it's a calculated column
#    }
  }

  return $cols;
}

sub header {
  my $self = shift;
  my $colid = shift;

  my $headers = $self->{headers};
#  return "ARRAY" unless (ref($headers) eq "ARRAY");
  my $toHeader = $self->{toHeader};
#  return "HASH" unless (ref($toHeader) eq "HASH");

  if (defined($colid)) {
    return $toHeader->{$colid};
  }

  return wantarray ? @$headers : undef;
}

sub url {
  my $self = shift;
  my ($cid, $aid) = @_;
  my $a2show = $self->{a2show};
  my @a2show = @{$a2show} if ref($a2show);
  my @urls;
  my %tourls;
  foreach (@a2show) {
    my $ci = $_->[1];
    my $pi = $_->[2];
    push @urls, $_->[4];
    $tourls{"$ci~-~$pi"} = $_->[4];
  }
  if (defined($cid) or defined($aid)) {
    @urls = ();
    push @urls, $tourls{"$cid~-~$aid"};
  }
  return wantarray ? @urls : $urls[0];
}

sub output {
  my $self = shift;
  my $output = shift;
  if ($output) {
    if ($output =~ /^(vText|hText|vHtml|hHtml|vExcel|hExcel|hashRef|XML)$/i) {
      $self->{output} = $output;
    } else {
      warn "output(): invalid output type ($output), output attribute not changed!";
      return;
    }
  }
  $self->{output};
}


=header2

  Title:    do_view
  Function: do the query set by the view settings, store the results, mark some attributes

=cut

sub do_view {

  my $self = shift;
  my $mcid = $self->mcid;
  my $mrefid = $self->mrefid;
  my $a2show = $self->a2show;

  my %vres = $self->xrt->relational_query(-mcid=>$mcid, -mrefids=>$mrefid, -a2show=>$a2show);

  my $num_entry = 0;
  my @entry_ids = sort(keys(%vres));
  $num_entry += @entry_ids;

  $self->{results} = \%vres;
  $self->{entry_ids} = \@entry_ids;
  $self->{outdated} = 0;

  return wantarray ? @entry_ids : $num_entry;

}

=header2

  Title:    getvalue
  Function: get value of a give cid and aid from the view results

=cut

sub getvalue {
  my $self = shift;

  $self->do_view if ($self->{outdated});

  my %vres = %{$self->{results}};
  my @mrefids = sort(keys(%vres));

  my ($mrefid, $refid, $cid, $class, $aid, $attribute) =
         rearrange([
                    [qw(MREFID MGREF)],
                    [qw(REFID GREF)],
                    'CID',
                    'CLASS',
                    'AID',
                    'ATTRIBUTE'
                   ],@_);

  $cid = 0 if ($class eq '0' or $class eq '-');
  $cid = $self->factory->class2cid($class) unless defined($cid);
  $cid = -1 if ($class && $cid eq undef);

  $aid ||= $self->factory->attribute2aid($attribute);
  $aid = -1 if ($attribute && !($aid));

  return if ($cid == -1 or $aid == -1);

  $mrefid ||= $mrefids[0];
  my $entry = $vres{$mrefid};
  return unless $entry;

  my @values;
  if ($refid) {
    @values = ref($entry->{"$refid~-~$cid~-~$aid"}) ? @{ $entry->{"$refid~-~$cid~-~$aid"} } : ();
  } else {
    my @ref_cid_aid = sort(keys(%{$entry}));
    foreach (@ref_cid_aid) {
      if (/^.+?~-~$cid~-~$aid$/) {
        @values = (@values, @{$entry->{$_}});
      }
    }
  }

  return wantarray ? @values : $values[0];

}


=header2 fetchentry

  Title:    fetchentry
  Function: fetch the view results (do do_view first if not been done) entry by entry in a defined format

=cut

sub fetchentry {
  my $self = shift;

  $self->do_view if ($self->{outdated});

  my $entry_id = shift @{$self->{entry_ids}};

  my $entry;

  if ($entry_id) {
    my %vres = %{$self->{results}};
    $entry = $self->_reformat($vres{$entry_id});
  } 

  return $entry;

}

=header2

  Title:     _reformat
  Function:  convert the view results to a few types of format
  Type:      private

=cut

sub _reformat {
  my $self = shift;
  my %entry = %{shift @_};
 
  my $output = $self->output;
  $output ||= 'hText';

  ################ generate the hash of array of grefs (%cid2refid), one related class could have more than one entry
  my %cid2refid = ();
  my %DONE = ();
  foreach (sort(keys(%entry))) {
    my @ids = split /~-~/;
    push @{ $cid2refid{$ids[1]} }, $ids[0] unless ($DONE{"$ids[0]-$ids[1]"}++);
  }

  my $a2show = $self->a2show;

  ################ generate the data structure -- %show
  my %show = ();
  my $max_row = 0;
  my %getColID = ();
  foreach (@{$a2show}) {
    my $colid = $_->[0];
    my $cid = $_->[1];
    my $aid = $_->[2];
    $getColID{"$cid~-~$aid"} = $colid if $aid =~ /^\d+$/;
    $getColID{"$cid~-~$colid"} = $colid;

    if ($aid =~ /^\d+$/) {  ### normal column
      ### we have to test $cid2refid{$cid} first, if one cid doesn't have any results, $cid2refid{$cid} will be a undefined varaible, so can not be a ref
      my @refid = ref($cid2refid{$cid}) eq "ARRAY" ? @{ $cid2refid{$cid} } : (); 

      $max_row = @refid unless ($max_row > @refid or $colid=~/^_/);
  
      foreach (@refid) {
        push @{ $show{"$colid"} }, $entry{"$_~-~$cid~-~$aid"};
      }
      $self->{col2value} = \%show;
    }
  }

  ##### for calculated columns
  foreach (@{$a2show}) {
    my $colid = $_->[0];
    my $cid = $_->[1];
    my $aid = $_->[2];

    if ($aid =~ /^sub/) {  ### calculated column
      my $func = eval($aid);
      my $value = &$func($self);
      @{ $show{"$colid"} } = ref($value) eq "ARRAY" ? @{$value} : (["$value"]);

      #### put the caculated value back to the $entry hash for later use
      my @refid = ref($cid2refid{$cid}) eq "ARRAY" ? @{ $cid2refid{$cid} } : (); 
      if (@refid == 1) {
        my $refid = $refid[0];
        foreach (@{$show{"$colid"}}) {
	  my $row = $_;
          foreach (@{$row}) {
	    push @{ $entry{"$refid~-~$cid~-~$colid"} }, $_;
	  }
	}
      } else {
        for (my $ii=0; $ii<@refid; $ii++) {
          $entry{"$refid[$ii]~-~$cid~-~$colid"} = $show{"$colid"}->[$ii] if ref($show{"$colid"}->[$ii]) eq "ARRAY";
        }
      }

      $max_row = @{ $show{"$colid"} } unless ($max_row > @{ $show{"$colid"} } or $colid=~/^_/);

      $self->{col2value} = \%show;
    }
  }

  ########### format the output:
  my $vres;

  if (uc($output) eq 'HTEXT') { ######## tab delimited text table 

    for (my $i=0; $i<$max_row; $i++) {  
      foreach (@{$a2show}) {
        unless ($_->[0] =~ /^_/) {
          my $colid = $_->[0];
          my @values =  ref($show{"$colid"}->[$i]) ? @{ $show{"$colid"}->[$i] } : ();
          $vres .= join('|', @values) . "\t";
        }
      }
      chop $vres;
      $vres .= "\n";
    }
    chomp $vres;

  } elsif (uc($output) eq 'HASHREF') {  ########### output an hash ref

    $show{max_row} = $max_row;
    $vres = \%show;

  } elsif (uc($output) eq 'VTEXT') {

  } elsif (uc($output) eq 'HHTML') {   ###### output a harizontal html table
#      $vres .= "  <tr>\n";
    foreach (@{$a2show}) {
      unless ($_->[0] =~ /^_/) {
        my $colid = $_->[0];
        my $URL = $_->[4];
        my @url = ();
        if ($URL =~ /^sub\s*?\{/) {  ####### 
          my $func = eval($URL);
          @url = &$func($self);  ### TODO: error handling needed when something wrong with the func
        } 

	$vres .= "    <td valign='top'>";

	my $ii = 0;
        for (my $i=0; $i<$max_row; $i++) {  
          my @values =  ref($show{"$colid"}->[$i]) ? @{ $show{$colid}->[$i] } : ();
          foreach (@values) {
              my $my_url = @url ? $url[$ii] : $URL;
	      my $url_encoded = uri_escape($_);
              $my_url =~ s/\*/$url_encoded/g;
              s/^(.+)$/<a target='' href='$my_url'>$1<\/a>/ if ($my_url);
	      $ii++;
          }
          my $tcell = @values ? join(', ', @values) : '&nbsp;';
          $tcell = '&nbsp;' if ($tcell=~/^\s*$/);
          $vres .= "$tcell<br>";
        }
        $vres =~ s/<br>$//;
        $vres .= "</td>\n";
      }
    }
    chomp $vres; 

  } elsif (uc($output) eq 'VHTML') {

  } elsif (uc($output) eq 'HEXCEL') {

  } elsif (uc($output) eq 'VEXCEL') {

  } elsif (uc($output) eq 'XML') {
    ### generate xml output for data in $entry
    my $indent = "   ";
    my $baseLevel = 1;
    my %cid2class = $self->xrt->cid2class;
    my @cids = sort keys %cid2refid ;  ### all classes
    my $mcid = $self->mcid;  ### main class id for this view
    for (my $ii=0; $ii<@cids; $ii++) {
      my $cid = $cids[$ii];
      my @aids = (); ### all attributes of the current class
      my %DONE = ();
      foreach (sort keys %entry) {
        my @ids = split /~-~/;
	push @aids, $ids[2] if ($ids[1] eq $cids[$ii] and !($DONE{$ids[2]}++));
      }

      my @refids = @{$cid2refid{$cid}};
      foreach (@refids) {
        my $refid = $_;
	my $class = $cid2class{$cid||$mcid};
        $vres .= ("$indent"x($baseLevel+($cid==0?0:1)))."<$class id=\"$refid\">\n";
        foreach (@aids) {
          my $aid = $_;
          my $colid = $getColID{"$cid~-~$aid"};
	  next if !defined($colid) or $colid=~/^_/; # should we hide the internal columns, maybe we need an option for that
          my $header = $self->header($colid) || "_undefined_attribute_";
	  $header =~ s/ /_/g;
          my @values = @{$entry{"$refid~-~$cid~-~$aid"}};
	  foreach (@values) {
            $vres .= ("$indent"x($baseLevel+($cid==0?1:2)))."<$header>$_</$header>\n";
	  }
        }
        $vres .= ("$indent"x($baseLevel+($cid==0?0:1)))."<\/$class>\n" if $cid;
      }
      $vres .= ("$indent"x$baseLevel)."<\/".$cid2class{$mcid}.">\n" if $ii==(@cids-1);
    }
    chomp $vres;

  } else {

  }

  return $vres;

}

sub show_view {
  my $self = shift;
  my $count = shift;
  my $hasCount = 1 if $count;

  my $output = $self->output;
  $output ||= 'hText';  

  my $outString = "";

  if (uc($output) eq 'HHTML') {
    $outString .= "<table border='1' cellpadding='2' cellspacing='0' width='100%' bordercolor='#808080' style='border-collapse: collapse'>\n";
    my @headers = $self->header;
    $outString .= "  <tr class='tab_header' bgcolor='#d9e4f8'>\n    ".($hasCount ? "<td><b>#</b></td>" : "");
    foreach (@headers) {
      $outString .= "    <td><b>$_</b></td>\n";
    }
    $outString .= "  </tr>\n";
    while (my $row = $self->fetchentry) {
      if ($hasCount) {
        $outString .= "  <tr class='tab_body'>\n    <td valign='top'>$count</td>\n$row\n  </tr>\n    ";
        $count++;
      } else {
        $outString .= "  <tr>\n$row\n  </tr>\n    ";
      }
    }
    $outString .= "</table>\n";
  
  } elsif (uc($output) eq 'XML') {
    $outString .= "<?xml version=\"1.0\"?>\n";
    $outString .= "<!-- This XML is generated by XRT TBrowse VERSION:$VERSION -->\n";
    $outString .= "<xrt dataset=\"".$self->mclass."\">\n";
    while (my $row = $self->fetchentry) {
      $outString .= $row . "\n";
    }
    $outString .= "</xrt>\n";

  } elsif (uc($output) eq 'HTEXT') {
    while (my $row = $self->fetchentry) {
      $outString .= $row . "\n";
    }
  }

  return $outString;
}


1;

__END__

=head1 AUTHOR

Junjun Zhang E<lt>junjun@genet.sickkids.on.caE<gt>.

Copyright (c) 2004 The Centre for Applied Genomics.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.  See DISCLAIMER for
disclaimers of warranty.

=cut
